`timescale 1ns/1ps

module testbench;
//-----------------------------------------------------------------------------
// Enable verification summary for console
// Warning! Verification may slow down simulation.
`define __VER
//-----------------------------------------------------------------------------

//========================================================================
// Array constants table  
//========================================================================
parameter numWLP  = 8;
parameter numWL   = 1024;
parameter numMux  = 256;
parameter numSect = 8;

parameter A_MSB = 21;
parameter A_SEC_MSB = 18;

parameter sizePage   = numWLP * numMux;
parameter sizeSector = numWL * numMux;
parameter sizeMem    = numSect * sizeSector;

//========================================================================
// Timing units table  
//========================================================================
parameter ns_unit =  1;
parameter us_unit =  1000;
parameter ms_unit =  1000 * 1000;
parameter sec_unit = 1000 * 1000 * 1000;

//========================================================================
// Timing table for SERIAL I/F  
//========================================================================
parameter	Tvcs = 100_000;	// Ucc setup time in ns

parameter Tcp    = 100;
parameter Tcp_er = 380;
parameter Tcp_wr = 140;
parameter Tsu_tck_tdi = 5;
parameter Th_tdi_tck = 0;
parameter Ta_tck = 35;
 
integer i, j, k;

//------------------------------------------------------------------------
// I/O Ports                                                            
//------------------------------------------------------------------------
reg STROBE; 
reg MRST; 
reg TCK; 
tri TDI; 
reg TDI_IN;

//------------------------------------------------------------------------
// Testbench routines                                                   
//------------------------------------------------------------------------
reg [15:0]  rdbuf;  // read  buffer
reg [15:0]  rdbufm; // read  buffer
reg [15:0]  wrbuf;	// write buffer

reg [A_MSB-1:0] ap; // address pointer
reg             drive; // data bus drive flags
reg             match; // match flag
reg             dismatch; // dismatch flag


integer test_no;

reg [7:0] mem [sizeMem-1:0]; // ethalon memory

assign TDI = drive ? TDI_IN : 1'bz;

//=============================================================================
//===== TOP Device =====
//=============================================================================
K1636RR4U dut (
  .nWE(1'b1),
  .nOE(1'b1),
  .nCE(1'b1),
  .A(21'b0),
  .D(),
  .A9_HV(),
  .OE_HV(),
  .TDI(TDI),
  .TCK(TCK),
  .STROBE(STROBE),
  .MRST(MRST),
  .SCK(),
  .SI(),
  .SEL_SPI(),
  .SO()
); 

//=============================================================================
//===== Initializing ports  =====
//=============================================================================
initial begin
  TDI_IN = 1'b0;
  TCK = 1'b0;
  STROBE = 1'b0;
  MRST = 1'b0;

  ap = {A_MSB{1'b0}};
  drive = 1'b0;
  match = 1'b1;
  dismatch = 1'b0;
end

//=============================================================================
//===== Test sequence itself =====
//=============================================================================
initial begin
  #(Tvcs); // Wait for model wakeup

  $display("RUN[TEST]: SERIAL FLASH I/F testing");

  MRST = 1'b1;

//=== Test1 - ERASE 1 ==========================================
  test_no=1;  
  ap = {3'h0, {A_SEC_MSB{1'b0}}};

  for (i = 0; i < numSect; i = i + 1) begin
    CMD_Erase_Sector (ap[A_MSB-1:A_SEC_MSB], 
                     0); // byte mode 

    ap[A_MSB-1:A_SEC_MSB] = ap[A_MSB-1:A_SEC_MSB] + 1;
  end

//=== Test2 - READ BYTE 1 ======================================
  test_no=2;  
  ap = {3'h0, {A_SEC_MSB{1'b0}}};

  for (i = 0; i < numSect*2; i = i + 1) begin
    CMD_Read (ap, 
         256, 
         0); // byte mode 

    ap[A_MSB-1:A_SEC_MSB-1] = ap[A_MSB-1:A_SEC_MSB-1] + 1;  
  end

//=== Test3 - PROGRAM BYTE ====================================
  test_no=3;  
  ap = {3'h0, {A_SEC_MSB{1'b0}}};


  for (i = 0; i < numSect*2; i = i + 1) begin
    for (j = 0; j < 256; j = j + 1) begin
      FillWrBuffer;
      if (j==0) begin
        CMD_Program (ap, 
                    wrbuf[7:0], 8'h00,
                    0, // byte mode 
                    0); // first write cycle (not repeat)
      end else begin 
        CMD_Program (ap, 
                    wrbuf[7:0], 8'h00, 
                    0, // byte mode 
                    1); // repeat write cycle
      end
      ap[A_SEC_MSB-2:0] = ap[A_SEC_MSB-2:0] + 1;
    end
    ap[A_MSB-1:A_SEC_MSB-1] = ap[A_MSB-1:A_SEC_MSB-1] + 1;  
    ap[A_SEC_MSB-2:0] = 0;
  end

//=== Test4 - READ BYTE 2 ======================================
  test_no=4;  
  ap = {3'h0, {A_SEC_MSB{1'b0}}};

  for (i = 0; i < numSect*2; i = i + 1) begin
    CMD_Read (ap, 
         256, 
         0); // byte mode 

    ap[A_MSB-1:A_SEC_MSB-1] = ap[A_MSB-1:A_SEC_MSB-1] + 1;  
  end

//=== Test5 - ERASE 2 ==========================================
  test_no=5;  
  ap = {3'h0, {A_SEC_MSB{1'b0}}};

  for (i = 0; i < numSect; i = i + 1) begin
    CMD_Erase_Sector (ap[A_MSB-1:A_SEC_MSB], 
                     1); // 16-bit mode

    ap[A_MSB-1:A_SEC_MSB] = ap[A_MSB-1:A_SEC_MSB] + 1;
  end

//=== Test6 - READ WORD 1 ======================================
  test_no=6;  
  ap = {3'h0, {A_SEC_MSB{1'b0}}};

  for (i = 0; i < numSect*2; i = i + 1) begin
    CMD_Read (ap, 
         256, 
         1); // 16-bit mode

    ap[A_MSB-1:A_SEC_MSB-1] = ap[A_MSB-1:A_SEC_MSB-1] + 1;  
  end

//=== Test7 - PROGRAM WORD ====================================
  test_no=7;  
  ap = {3'h0, {A_SEC_MSB{1'b0}}};


  for (i = 0; i < numSect*2; i = i + 1) begin
    for (j = 0; j < 256; j = j + 2) begin
      FillWrBuffer;
      if (j==0) begin
        CMD_Program (ap, 
                    wrbuf[7:0], wrbuf[15:8], 
                    1, // 16-bit mode
                    0); // first write cycle (not repeat) 
      end else begin 
        CMD_Program (ap, 
                    wrbuf[7:0], wrbuf[15:8], 
                    1, // 16-bit mode
                    1); // repeat write cycle
      end
      ap[A_SEC_MSB-2:0] = ap[A_SEC_MSB-2:0] + 2;
    end
    ap[A_MSB-1:A_SEC_MSB-1] = ap[A_MSB-1:A_SEC_MSB-1] + 1;  
    ap[A_SEC_MSB-2:0] = 0;
  end

//=== Test8 - READ WORD 2 ======================================
  test_no=8;  
  ap = {3'h0, {A_SEC_MSB{1'b0}}};

  for (i = 0; i < numSect*2; i = i + 1) begin
    CMD_Read (ap, 
         256, 
         1); // 16-bit mode

    ap[A_MSB-1:A_SEC_MSB-1] = ap[A_MSB-1:A_SEC_MSB-1] + 1;  
  end
//===================================================================
  `ifdef __VER
  if (dismatch) $display("ERROR[TEST]: Some operations unsuccessfully finished or read data is not matched with ethalon!");
  else $display("DONE[TEST]: All operations successfully finished and data is matched with ethalon");
  `else
  $display("DONE[TEST]: All operations successfully finished");
  `endif
  
  $display("\n---< See ya later, allagator! >---\n");


  $finish;
end

always @ (negedge match) dismatch = 1; // dismatch test detector

//=============================================================================
//===== TASKS =====
//=============================================================================
// Fills write buffer with random numbers.
task FillWrBuffer;
begin
  wrbuf=$random;
end
endtask

//=============================================================================
//===== SERIAL FLASH I/F routines =====
//=============================================================================
// Strobe cycle
task Strobe_cycle;
begin
  fork
    #(Tcp/2) TCK = 1'b1;
    #(Tcp)   TCK = 1'b0;
  join

  STROBE = 1'b1;

  fork
    #(Tcp/2) TCK = 1'b1;
    #(Tcp)   TCK = 1'b0;
  join

  STROBE = 1'b0;
end
endtask

// Full cycle byte mode
task Full_cycle_byte;
input [10:0] cmd;
input [16:0] addr;
input [7:0]  data;
input time Tcp;

reg [1:0] start_symbol;
integer i;

reg [7:0] data_reg;

begin
  data_reg = data;

  start_symbol = 2'b00;
  drive = 1'b1;
  for (i = 0; i < 2; i = i + 1) begin
    fork
      #((Tcp/2)-Tsu_tck_tdi) TDI_IN = start_symbol[i];
      //#((Tcp/2)+Th_tdi_tck) TDI_IN = 1'bx;
      #(Tcp/2) TCK = 1'b1;
      #(Tcp)   TCK = 1'b0; 
    join
  end
  for (i = 0; i < 11; i = i + 1) begin
    fork
      #((Tcp/2)-Tsu_tck_tdi) TDI_IN = cmd[i];
      //#((Tcp/2)+Th_tdi_tck) TDI_IN = 1'bx;
      #(Tcp/2) TCK = 1'b1;
      #(Tcp)   TCK = 1'b0; 
    join
  end
  for (i = 0; i < 17; i = i + 1) begin
    fork
      #((Tcp/2)-Tsu_tck_tdi) TDI_IN = addr[i];
      //#((Tcp/2)+Th_tdi_tck) TDI_IN = 1'bx;
      #(Tcp/2) TCK = 1'b1;
      #(Tcp)   TCK = 1'b0; 
    join
  end
  for (i = 0; i < 8; i = i + 1) begin
    fork
      #((Tcp/2)-Tsu_tck_tdi) TDI_IN = data_reg[i];
      //#((Tcp/2)+Th_tdi_tck) TDI_IN = 1'bx;
      #(Tcp/2) TCK = 1'b1;
      #(Tcp)   TCK = 1'b0; 
    join
  end
  drive = 1'b0;
  for (i = 0; i < 3; i = i + 1) begin
    fork
      #(Tcp/2) TCK = 1'b1;
      #(Tcp)   TCK = 1'b0;
    join
  end
end
endtask

// Full cycle nbyte mode
task Full_cycle_nbyte;
input [10:0] cmd;
input [15:0] addr;
input [15:0] data;
input time Tcp;

reg [1:0] start_symbol;
integer i;

reg [15:0] data_reg;

begin
  data_reg = data;

  start_symbol = 2'b00;
  drive = 1'b1;
  for (i = 0; i < 2; i = i + 1) begin
    fork
      #((Tcp/2)-Tsu_tck_tdi) TDI_IN = start_symbol[i];
      //#((Tcp/2)+Th_tdi_tck) TDI_IN = 1'bx;
      #(Tcp/2) TCK = 1'b1;
      #(Tcp)   TCK = 1'b0; 
    join
  end
  for (i = 0; i < 11; i = i + 1) begin
    fork
      #((Tcp/2)-Tsu_tck_tdi) TDI_IN = cmd[i];
      //#((Tcp/2)+Th_tdi_tck) TDI_IN = 1'bx;
      #(Tcp/2) TCK = 1'b1;
      #(Tcp)   TCK = 1'b0; 
    join
  end
  for (i = 0; i < 16; i = i + 1) begin
    fork
      #((Tcp/2)-Tsu_tck_tdi) TDI_IN = addr[i];
      //#((Tcp/2)+Th_tdi_tck) TDI_IN = 1'bx;
      #(Tcp/2) TCK = 1'b1;
      #(Tcp)   TCK = 1'b0; 
    join
  end
  for (i = 0; i < 16; i = i + 1) begin
    fork
      #((Tcp/2)-Tsu_tck_tdi) TDI_IN = data_reg[i];
      //#((Tcp/2)+Th_tdi_tck) TDI_IN = 1'bx;
      #(Tcp/2) TCK = 1'b1;
      #(Tcp)   TCK = 1'b0; 
    join
  end
  drive = 1'b0;
  for (i = 0; i < 3; i = i + 1) begin
    fork
      #(Tcp/2) TCK = 1'b1;
      #(Tcp)   TCK = 1'b0;
    join
  end
end
endtask

// Command cycle
task Cmd_cycle;
input [10:0] cmd;
input [11:0] addr;
input [7:0] data;
input time Tcp;

reg [1:0] start_symbol;
integer i;
begin
  start_symbol = 2'b10;
  drive = 1'b1;
  for (i = 0; i < 2; i = i + 1) begin
    fork
      #((Tcp/2)-Tsu_tck_tdi) TDI_IN = start_symbol[i];
      //#((Tcp/2)+Th_tdi_tck) TDI_IN = 1'bx;
      #(Tcp/2) TCK = 1'b1;
      #(Tcp)   TCK = 1'b0; 
    join
  end
  for (i = 0; i < 11; i = i + 1) begin
    fork
      #((Tcp/2)-Tsu_tck_tdi) TDI_IN = cmd[i];
      //#((Tcp/2)+Th_tdi_tck) TDI_IN = 1'bx;
      #(Tcp/2) TCK = 1'b1;
      #(Tcp)   TCK = 1'b0; 
    join
  end
  for (i = 0; i < 12; i = i + 1) begin
    fork
      #((Tcp/2)-Tsu_tck_tdi) TDI_IN = addr[i];
      //#((Tcp/2)+Th_tdi_tck) TDI_IN = 1'bx;
      #(Tcp/2) TCK = 1'b1;
      #(Tcp)   TCK = 1'b0; 
    join
  end
  for (i = 0; i < 8; i = i + 1) begin
    fork
      #((Tcp/2)-Tsu_tck_tdi) TDI_IN = data[i];
      //#((Tcp/2)+Th_tdi_tck) TDI_IN = 1'bx;
      #(Tcp/2) TCK = 1'b1;
      #(Tcp)   TCK = 1'b0; 
    join
  end
  drive = 1'b0;
  for (i = 0; i < 3; i = i + 1) begin
    fork
      #(Tcp/2) TCK = 1'b1;
      #(Tcp)   TCK = 1'b0; 
    join
  end
end
endtask

// Read cycle byte
task Read_cycle_byte;
input [10:0] cmd;
input [16:0] addr;
output reg [7:0] data;
input time Tcp;

reg [1:0] start_symbol;
reg [7:0] data_reg;

integer i;

begin
  start_symbol = 2'b01;
  drive = 1'b1;
  for (i = 0; i < 2; i = i + 1) begin
    fork
      #((Tcp/2)-Tsu_tck_tdi) TDI_IN = start_symbol[i];
      //#((Tcp/2)+Th_tdi_tck) TDI_IN = 1'bx;
      #(Tcp/2) TCK = 1'b1;
      #(Tcp)   TCK = 1'b0; 
    join
  end
  for (i = 0; i < 11; i = i + 1) begin
    fork
      #((Tcp/2)-Tsu_tck_tdi) TDI_IN = cmd[i];
      //#((Tcp/2)+Th_tdi_tck) TDI_IN = 1'bx;
      #(Tcp/2) TCK = 1'b1;
      #(Tcp)   TCK = 1'b0; 
    join
  end
  for (i = 0; i < 17; i = i + 1) begin
    fork
      #((Tcp/2)-Tsu_tck_tdi) TDI_IN = addr[i];
      //#((Tcp/2)+Th_tdi_tck) TDI_IN = 1'bx;
      #(Tcp/2) TCK = 1'b1;
      #(Tcp)   TCK = 1'b0; 
    join
  end
  drive = 1'b0;
  fork
    #(Tcp/2) TCK = 1'b1;
    #(Tcp)   TCK = 1'b0; 
  join
  for (i = 0; i < 8; i = i + 1) begin
    fork
      #(Ta_tck+1)  data_reg[i] = TDI;
      #(Tcp/2) TCK = 1'b1;
      #(Tcp)   TCK = 1'b0; 
    join
  end

  data = data_reg;

end
endtask

// Read cycle nbyte
task Read_cycle_nbyte;
input [10:0] cmd;
input [15:0] addr;
output reg [15:0] data;
input time Tcp;

reg [1:0] start_symbol;
reg [15:0] data_reg;

integer i;

begin
  start_symbol = 2'b01;
  drive = 1'b1;
  for (i = 0; i < 2; i = i + 1) begin
    fork
      #((Tcp/2)-Tsu_tck_tdi) TDI_IN = start_symbol[i];
      //#((Tcp/2)+Th_tdi_tck) TDI_IN = 1'bx;
      #(Tcp/2) TCK = 1'b1;
      #(Tcp)   TCK = 1'b0; 
    join
  end
  for (i = 0; i < 11; i = i + 1) begin
    fork
      #((Tcp/2)-Tsu_tck_tdi) TDI_IN = cmd[i];
      //#((Tcp/2)+Th_tdi_tck) TDI_IN = 1'bx;
      #(Tcp/2) TCK = 1'b1;
      #(Tcp)   TCK = 1'b0; 
    join
  end
  for (i = 0; i < 16; i = i + 1) begin
    fork
      #((Tcp/2)-Tsu_tck_tdi) TDI_IN = addr[i];
      //#((Tcp/2)+Th_tdi_tck) TDI_IN = 1'bx;
      #(Tcp/2) TCK = 1'b1;
      #(Tcp)   TCK = 1'b0; 
    join
  end
  drive = 1'b0;
  fork
    #(Tcp/2) TCK = 1'b1;
    #(Tcp)   TCK = 1'b0;
  join
  for (i = 0; i < 16; i = i + 1) begin
    fork
      #(Ta_tck+1)  data_reg[i] = TDI;
      #(Tcp/2) TCK = 1'b1;
      #(Tcp)   TCK = 1'b0; 
    join
  end

  data = data_reg;

end
endtask

// Read repeat byte
task Read_repeat_byte;
output reg [7:0] data;
input time Tcp;

reg [1:0] start_symbol;
reg [7:0] data_reg;

integer i;

begin
  start_symbol = 2'b11;
  drive = 1'b1;
  for (i = 0; i < 2; i = i + 1) begin
    fork
      #((Tcp/2)-Tsu_tck_tdi) TDI_IN = start_symbol[i];
      //#((Tcp/2)+Th_tdi_tck) TDI_IN = 1'bx;
      #(Tcp/2) TCK = 1'b1;
      #(Tcp)   TCK = 1'b0; 
    join
  end
  drive = 1'b0;
  for (i = 0; i < 8; i = i + 1) begin
    fork
      #(Ta_tck+1)  data_reg[i] = TDI;
      #(Tcp/2) TCK = 1'b1;
      #(Tcp)   TCK = 1'b0; 
    join
  end

  data = data_reg;
  
end
endtask

// Read repeat nbyte
task Read_repeat_nbyte;
output reg [15:0] data;
input time Tcp;

reg [1:0] start_symbol;
reg [15:0] data_reg;

integer i;

begin
  start_symbol = 2'b11;
  drive = 1'b1;
  for (i = 0; i < 2; i = i + 1) begin
    fork
      #((Tcp/2)-Tsu_tck_tdi) TDI_IN = start_symbol[i];
      //#((Tcp/2)+Th_tdi_tck) TDI_IN = 1'bx;
      #(Tcp/2) TCK = 1'b1;
      #(Tcp)   TCK = 1'b0; 
    join
  end
  drive = 1'b0;
  for (i = 0; i < 16; i = i + 1) begin
    fork
      #(Ta_tck+1)  data_reg[i] = TDI;
      #(Tcp/2) TCK = 1'b1;
      #(Tcp)   TCK = 1'b0; 
    join
  end

  data = data_reg;

end
endtask


// Write repeat byte
task Write_repeat_byte;
input [7:0] data;
input time Tcp;

reg [1:0] start_symbol;
reg [15:0] data_reg;

integer i;

begin
  data_reg = data;

  start_symbol = 2'b11;
  drive = 1'b1;
  for (i = 0; i < 2; i = i + 1) begin
    fork
      #((Tcp/2)-Tsu_tck_tdi) TDI_IN = start_symbol[i];
      //#((Tcp/2)+Th_tdi_tck) TDI_IN = 1'bx;
      #(Tcp/2) TCK = 1'b1;
      #(Tcp)   TCK = 1'b0; 
    join
  end
  for (i = 0; i < 8; i = i + 1) begin
    fork
      #((Tcp/2)-Tsu_tck_tdi) TDI_IN = data_reg[i];
      //#((Tcp/2)+Th_tdi_tck) TDI_IN = 1'bx;
      #(Tcp/2) TCK = 1'b1;
      #(Tcp)   TCK = 1'b0; 
    join
  end
  drive = 1'b0;
  for (i = 0; i < 3; i = i + 1) begin
    fork
      #(Tcp/2) TCK = 1'b1;
      #(Tcp)   TCK = 1'b0;
    join
  end
end
endtask

// Write repeat nbyte
task Write_repeat_nbyte;
input [15:0] data;
input time Tcp;

reg [1:0] start_symbol;
reg [15:0] data_reg;

integer i;

begin
  data_reg = data;

  start_symbol = 2'b11;
  drive = 1'b1;
  for (i = 0; i < 2; i = i + 1) begin
    fork
      #((Tcp/2)-Tsu_tck_tdi) TDI_IN = start_symbol[i];
      //#((Tcp/2)+Th_tdi_tck) TDI_IN = 1'bx;
      #(Tcp/2) TCK = 1'b1;
      #(Tcp)   TCK = 1'b0; 
    join
  end
  for (i = 0; i < 16; i = i + 1) begin
    fork
      #((Tcp/2)-Tsu_tck_tdi) TDI_IN = data_reg[i];
      //#((Tcp/2)+Th_tdi_tck) TDI_IN = 1'bx;
      #(Tcp/2) TCK = 1'b1;
      #(Tcp)   TCK = 1'b0; 
    join
  end
  drive = 1'b0;
  for (i = 0; i < 3; i = i + 1) begin
    fork
      #(Tcp/2) TCK = 1'b1;
      #(Tcp)   TCK = 1'b0;
    join
  end
end
endtask


// Read busy flag
task Read_busy_flag;
input time Tcp;
reg flag;
integer i;
integer n;
begin
  n = 10;
  flag = 1'b1;
  drive = 1'b0;
  for (i = 0; i < n; i = i + 1) begin
    fork
      #(Ta_tck+1)  flag = TDI;
      #(Tcp/2) TCK = 1'b1;
      #(Tcp)   TCK = 1'b0; 
    join

    if (flag) n = n - 1;
    else n = n + 1;
  end
end
endtask

//=============================================================================
//===== SERIAL FLASH I/F commands =====
//=============================================================================

// SECTOR ERASE
task CMD_Erase_Sector;
input [A_MSB-1:A_SEC_MSB] SA;
input nBYTE;
integer i;
begin
  Strobe_cycle;
  Cmd_cycle ({SA, 1'b0, 1'b0, 1'b1, 1'b1, 1'b0, 1'b1, 1'b0, 1'b0}, 
            12'h555, 
            8'hAA, 
            Tcp);

  Strobe_cycle;
  Cmd_cycle ({SA, 1'b0, 1'b0, 1'b1, 1'b1, 1'b0, 1'b1, 1'b0, 1'b0}, 
            12'hAAA, 
            8'h55, 
            Tcp);

  Strobe_cycle;
  Cmd_cycle ({SA, 1'b0, 1'b0, 1'b1, 1'b1, 1'b0, 1'b1, 1'b0, 1'b0}, 
            12'h555, 
            8'h80, 
            Tcp);

  Strobe_cycle;
  Cmd_cycle ({SA, 1'b0, 1'b0, 1'b1, 1'b1, 1'b0, 1'b1, 1'b0, 1'b0}, 
            12'h555, 
            8'hAA, 
            Tcp);

  Strobe_cycle;
  Cmd_cycle ({SA, 1'b0, 1'b0, 1'b1, 1'b1, 1'b0, 1'b1, 1'b0, 1'b0}, 
            12'hAAA, 
            8'h55, 
            Tcp);

  Strobe_cycle;
  if (!nBYTE) begin
    Full_cycle_byte ({SA, 1'b0, 1'b0, 1'b1, 1'b1, 1'b0, 1'b1, 1'b0, 1'b0}, 
                    17'h00555, 
                    8'h10, 
                    Tcp);
  end else begin
    Full_cycle_nbyte ({SA, 1'b0, 1'b0, 1'b1, 1'b1, 1'b0, 1'b1, 1'b1, 1'b0}, 
                     16'h00555, 
                     16'h0010, 
                     Tcp);
  end

  `ifdef __VER
  $display("RUN[Erase Sector]: Sector %d", SA);
  `endif

  Read_busy_flag(Tcp_er);

  `ifdef __VER
  $display("DONE[Erase Sector]: Operation finish, please verify it by read =)"); 
  for (i=0; i<sizeSector; i=i+1) mem[{SA,{A_SEC_MSB{1'b0}}}+i]<=8'hFF;
  `endif
end
endtask

// READ
task CMD_Read;
input [A_MSB-1:0] addr;
input [31:0] rdbytes;
input nBYTE;

reg [31:0] rdbytes_int;   // actual number of bytes to read
reg [A_MSB-1:0] addr_int; // actual address

integer j;

begin
  match = 1'b1;
  rdbytes_int   = rdbytes;
  addr_int = addr;

  if (!nBYTE) begin
    for (j = 0; j < rdbytes_int-1; j = j + 1) begin
      Strobe_cycle;
      if (j==0) Read_cycle_byte ({addr_int[A_MSB-1:A_SEC_MSB-1], 1'b0, 1'b1, 1'b0, 1'b1, 1'b1, 1'b0, 1'b0}, 
                                addr_int[A_SEC_MSB-2:0], 
                                rdbuf[7:0], 
                                Tcp);
      else Read_repeat_byte (rdbuf[7:0], 
                            Tcp);

      `ifdef __VER
      match = 1'b1;

      rdbufm[7:0]=mem[addr_int];

      if (rdbuf[7:0]==rdbufm[7:0]) begin
        //$display("DONE[SCH vs Ethalone]: rdbuf(%h)==rdbufm(%h), MATCH DATA =)",rdbuf,rdbufm); 
      end else begin
        match = 1'b0;
        $display("ERROR[SCH vs Ethalone]: rdbuf(%h)!=rdbufm(%h), addr=%h, DISMATCH DATA =( ", rdbuf[7:0], rdbufm[7:0], addr_int);
      end
      `endif

      addr_int[A_SEC_MSB-2:0] = addr_int[A_SEC_MSB-2:0] + 1;

    end
  end else begin
    for (j = 0; j < rdbytes_int-1; j = j + 2) begin
      Strobe_cycle;
      if (j==0) Read_cycle_nbyte ({addr_int[A_MSB-1:A_SEC_MSB-1], 1'b0, 1'b1, 1'b0, 1'b1, 1'b1, 1'b1, 1'b0}, 
                                 addr_int[A_SEC_MSB-2:1], 
                                 rdbuf, 
                                 Tcp);
      else Read_repeat_nbyte (rdbuf, 
                             Tcp);

      `ifdef __VER
      match = 1'b1;

      rdbufm={mem[addr_int+1], mem[addr_int]};

      if (rdbuf==rdbufm) begin
        //$display("DONE[SCH vs Ethalone]: rdbuf(%h)==rdbufm(%h), MATCH DATA =)",rdbuf,rdbufm); 
      end else begin
        match = 1'b0;
        $display("ERROR[SCH vs Ethalone]: rdbuf(%h)!=rdbufm(%h), addr=%h, DISMATCH DATA =( ",rdbuf,rdbufm,(addr_int));
      end
      `endif

      addr_int[A_SEC_MSB-2:0] = addr_int[A_SEC_MSB-2:0] + 2;

    end
  end
end
endtask

// PROGRAM BYTE / WORD
task CMD_Program;
input [A_MSB-1:0] ADDR;
input [7:0] DATA0;
input [7:0] DATA1;
input nBYTE;
input rep_eat;
begin
  if (!rep_eat) begin
    Strobe_cycle;
    Cmd_cycle ({ADDR[A_MSB-1:A_SEC_MSB-1], 1'b0, 1'b1, 1'b1, 1'b0, 1'b1, 1'b0, 1'b0}, 
              12'h555, 
              8'hAA, 
              Tcp);

    Strobe_cycle;
    Cmd_cycle ({ADDR[A_MSB-1:A_SEC_MSB-1], 1'b0, 1'b1, 1'b1, 1'b0, 1'b1, 1'b0, 1'b0}, 
              12'hAAA, 
              8'h55, 
              Tcp);
    
    Strobe_cycle;
    Cmd_cycle ({ADDR[A_MSB-1:A_SEC_MSB-1], 1'b0, 1'b1, 1'b1, 1'b0, 1'b1, 1'b0, 1'b0}, 
              12'h555, 
              8'hA0, 
              Tcp);

    Strobe_cycle;
    if (!nBYTE) begin
      Full_cycle_byte ({ADDR[A_MSB-1:A_SEC_MSB-1], 1'b0, 1'b1, 1'b1, 1'b0, 1'b1, 1'b0, 1'b0}, 
                      ADDR[A_SEC_MSB-2:0], 
                      DATA0, 
                      Tcp);
    end else begin
      Full_cycle_nbyte ({ADDR[A_MSB-1:A_SEC_MSB-1], 1'b0, 1'b1, 1'b1, 1'b0, 1'b1, 1'b1, 1'b0}, 
                       ADDR[A_SEC_MSB-2:1], 
                       {DATA1, DATA0}, 
                       Tcp);
    end
  end else begin
    Strobe_cycle;
    if (!nBYTE) begin
      Write_repeat_byte (DATA0, 
                        Tcp);
    end else begin
      Write_repeat_nbyte ({DATA1, DATA0}, 
                         Tcp);
    end
  end

  `ifdef __VER
  if (!nBYTE) $display("RUN[Program]: addr=%h, data=%h", ADDR, DATA0); 
  else $display("RUN[Program]: addr=%h, data=%h", ADDR, {DATA1, DATA0}); 
  `endif

  Read_busy_flag(Tcp_wr);

  `ifdef __VER
  $display("DONE[Program]: Operation finish, please verify it by read =)"); 
  if (!nBYTE) mem[ADDR]=DATA0;
  else {mem[ADDR+1], mem[ADDR]}={DATA1, DATA0};
  `endif

end
endtask

endmodule